using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;

public class DotMovement : MonoBehaviour
{
    public float speed = 2f; // Adjust the speed value to control the dot's movement speed
    public LineRenderer lineRenderer; // Reference to the LineRenderer component
    public GameObject capsule;
    public Transform sphereCenter; // Reference to the sphere's center transform
    public float lineOffset = 0.1f; // Offset from the sphere's surface
    public TextMeshProUGUI distanceText; // Reference to the UI TextMeshProUGUI component
    public Vector3 startPosition; // Serialized field for custom start position
    public GameObject discPrefab; // Reference to the disc prefab

    public KeyCode toggleRotationKey = KeyCode.Space; // Key to toggle rotation, mappable in the Inspector

    private float radius;
    private float timeSinceLastSphere = 0f;
    private Camera mainCamera; // Reference to the main camera
    private Vector3 uiTextPosition; // Position of the UI text in screen coordinates
    private Quaternion capsuleInitialRotation;
    private GameObject disc; // Reference to the instantiated disc object
    private Quaternion discInitialRotation;
    private bool isRotationEnabled = true; // Flag to track the rotation state
    private bool isMoving = false; // Flag to track if the dot is moving

    public bool IsMoving => isMoving; // Property to access the movement state

    private void Start()
    {
        radius = transform.parent.GetComponent<SphereCollider>().radius; // Get the radius of the sphere
        transform.position = startPosition; // Set the dot's position to the custom start position
        mainCamera = Camera.main; // Get reference to the main camera

        // Calculate and set the initial position of the UI text
        Vector3 surfacePosition = sphereCenter.position + (transform.position - sphereCenter.position).normalized * (radius + lineOffset);
        uiTextPosition = mainCamera.WorldToScreenPoint(surfacePosition);
        distanceText.rectTransform.position = uiTextPosition;

        // Instantiate the disc prefab
        disc = Instantiate(discPrefab, surfacePosition, Quaternion.identity, transform);
        disc.transform.up = transform.up; // Make the disc face the same direction as the dot's up vector   

        // Store the initial rotation of the capsule
        capsuleInitialRotation = capsule.transform.rotation;

        // Store the initial rotation of the disc
        discInitialRotation = disc.transform.rotation;
        discInitialRotation *= Quaternion.Euler(90f, 0f, 0f); // Apply a 90-degree rotation around the transform.right axis
    }

    private void Update()
    {
        float horizontalInput = Input.GetAxis("Horizontal");
        float verticalInput = Input.GetAxis("Vertical");

        Vector3 movement = new Vector3(horizontalInput, verticalInput, 0f);
        movement = movement.normalized * speed * Time.deltaTime * 0.1f; // Adjust the multiplication factor to control the dot's speed

        transform.position += movement; // Move the dot

        // Rotate the capsule to face the movement direction
        if (movement != Vector3.zero)
        {
            Quaternion targetRotation = Quaternion.LookRotation(movement, Vector3.up);
            capsule.transform.rotation = capsuleInitialRotation * targetRotation;
            isMoving = true; // Set the movement state to true
        }
        else
        {
            isMoving = false; // Set the movement state to false
        }

        // Rotate the sphere with Q and E keys
        float rotationInput = Input.GetAxis("Rotation");
        float rotationSpeed = 20f; // Adjust the rotation speed as needed

        // Apply rotation based on input
        transform.parent.Rotate(Vector3.up, rotationInput * rotationSpeed * Time.deltaTime);

        Vector3 sphereCenter = transform.parent.position;
        Vector3 dotToCenter = transform.position - sphereCenter;
        float distanceToCenter = dotToCenter.magnitude;

        if (distanceToCenter > radius)
        {
            Vector3 newDotPosition = sphereCenter + dotToCenter.normalized * radius;
            transform.position = newDotPosition; // Wrap the dot around the surface of the sphere
        }

        Vector3 surfacePosition = sphereCenter + dotToCenter.normalized * (radius + lineOffset);
        lineRenderer.SetPosition(0, transform.position);
        lineRenderer.SetPosition(1, surfacePosition); // Update the LineRenderer positions to draw the line to the surface

        float distanceToSurface = radius - distanceToCenter;
        float distanceInMiles = Mathf.Ceil(distanceToSurface * 7917.5f); // Calculate the distance in miles from the dot to the surface

        distanceText.SetText("Surface: " + distanceInMiles.ToString("0") + " miles");

        // Update the position of the disc to the surface position
        disc.transform.position = surfacePosition;

        // Calculate the direction vector from the surface position to the dot's position
        Vector3 lineDirection = transform.position - surfacePosition;

        // Calculate the rotation required to align the disc with the line direction
        Quaternion rotationTarget = Quaternion.LookRotation(lineDirection, transform.up);

        // Apply the rotation to the disc
        disc.transform.rotation = rotationTarget * discInitialRotation;

        // Update the UI text position if the dot or camera moves
        Vector3 offset = new Vector3(120f, 0f, 0f); // Adjust the offset as needed
        Vector3 currentUiTextPosition = mainCamera.WorldToScreenPoint(surfacePosition) + offset;
        if (currentUiTextPosition != uiTextPosition)
        {
            uiTextPosition = currentUiTextPosition;
            distanceText.rectTransform.position = uiTextPosition;
        }

        // Toggle rotation on button press
        if (Input.GetKeyDown(toggleRotationKey))
        {
            isRotationEnabled = !isRotationEnabled; // Toggle the rotation state
        }

        // Rotate the sphere continuously if enabled
        if (isRotationEnabled)
        {
            // Rotate the sphere continuously
            float rotationSpeedContinuous = 20f; // Adjust the continuous rotation speed as needed
            transform.parent.Rotate(Vector3.up, rotationSpeedContinuous * Time.deltaTime);
        }



    }
}
